/*--------------------------------------------------------------------------*\

    FILE....: FIFO.CPP
    TYPE....: C++ Functions
    AUTHOR..: David Rowe
    DATE....: 19/11/97

    Functions used to implement First In First Out (FIFO) queues of 16 bit
    words.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "fifo.h"
#include "wobbly.h"

/*--------------------------------------------------------------------------*\

								DEFINES

\*--------------------------------------------------------------------------*/

#define	FAIL	1		// returned by CheckFifo()

/*--------------------------------------------------------------------------*\

								CLASS

\*--------------------------------------------------------------------------*/

class FifoData {
    word   *pstart;		// first word in FIFO			
    word   *pend;		// one after last word in FIFO		
    word   *pwr;		// write pointer			
    word   *prd; 		// read pointer				
    USHORT size;		// total storage in FIFO
public:
	FifoData(USHORT size);
	~FifoData();
	int Write(word *buf, USHORT size);
	int Read(word *buf, USHORT size);
	void HowFull(USHORT *words);
	int CheckFifo();
};

/*--------------------------------------------------------------------------*\

							FIFO MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: Fifo::Fifo
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Creates a FIFO.

\*--------------------------------------------------------------------------*/

Fifo::Fifo(USHORT sz)
//  USHORT sz;		size of FIFO in words				 
{
    d = new FifoData(sz);
	if (d == NULL)
		throw Wobbly(FIFO_CANT_ALLOCATE_MEMORY);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Fifo::~Fifo
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Destorys a FIFO.

\*--------------------------------------------------------------------------*/

Fifo::~Fifo()
{
    delete d;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Fifo::Write
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Writes a block of words to a FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int Fifo::Write(word *buf, USHORT size)
//  word   *buf;	buffer of words to write to FIFO	
//  USHORT size;	size of FIFO in words				 
{
    return(d->Write(buf, size));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Fifo::Read
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Reads a block of words from a FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int Fifo::Read(word *buf, USHORT size)
//  word   *buf;	buffer of words read from FIFO		
//  USHORT size;	size of FIFO in words				 
{
    return(d->Read(buf, size));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Fifo::HowFull
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Determines number of used words in FIFO.

\*--------------------------------------------------------------------------*/

void Fifo::HowFull(USHORT *words)
//  word   *words;	number used words in FIFO			 
{
    d->HowFull(words);
}

/*--------------------------------------------------------------------------*\

						  FIFODATA MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoData::FifoData
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Creates a FIFO.

\*--------------------------------------------------------------------------*/

FifoData::FifoData(USHORT sz)
//  USHORT sz;		size of FIFO in words				 
{
    size = sz;
	pstart = new word[size];
	
    if (pstart == NULL)
		throw Wobbly(FIFO_CANT_ALLOCATE_MEMORY);
    pend = pstart + size;
	
    pwr = pstart;
    prd = pstart;
    size = sz;
    assert(CheckFifo() == OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoData::~FifoData
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Destorys a FIFO.

\*--------------------------------------------------------------------------*/

FifoData::~FifoData()
{
    delete [] pstart;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoData::Write
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Writes a block of words to a FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int FifoData::Write(word *buf, USHORT sz)
//  word   *buf;	buffer of words to write to FIFO	
//  USHORT sz;		size of buffer in words				 
{
    int		words_used;	// used space in FIFO		
    int		words_free;	// free space in FIFO		
    int		copy_first;	// size of first block move	
    word	*new_pwr;	// modified pwr				 

    // validate arguments

    assert(buf != NULL);
    assert(CheckFifo() == OK);

    // determine if there is data in FIFO to fill buf

    if (pwr >= prd)
		words_used = pwr - prd;
    if (prd > pwr)
		words_used = size - (prd - pwr);
	words_free = size - words_used - 1;
    if (words_free < sz)
		return(FIFO_FULL);

    // If buf overlaps end of linear array split into two block moves 

	if ((pwr + sz) > pend) {
		copy_first = (pend-pwr);

		memcpy(pwr, buf, copy_first*sizeof(word));
		memcpy(pstart, &buf[copy_first], (sz-copy_first)*sizeof(word));
    }
    else {
		memcpy(pwr, buf, sz*sizeof(word));
	}

    // increment pwr and wrap around if required 

    new_pwr = pwr + sz;
    if (new_pwr >= pend)
		pwr = pstart + (new_pwr - pend);
    else
		pwr = new_pwr;

    assert(CheckFifo() == OK);
    return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoData::Read
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Reads a block of words from a FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int FifoData::Read(word *buf, USHORT sz)
//  word   *buf;	buffer of words read from FIFO		
//  USHORT sz;		size of FIFO in words				 
{
    int		copy_first;	// size of first copy				 
    int		words_used;	// used space in FIFO				 
    word	*new_prd;	// modified prd					 

    /* validate arguments */

    assert(buf != NULL);
    assert(CheckFifo() == OK);

    // determine if there is enough data in FIFO to fill buf 

    if (pwr >= prd)
		words_used = pwr - prd;
  
    if (prd > pwr)
		words_used = size - (prd - pwr);
    if (words_used < sz)
		return(FIFO_EMPTY);

    // If buf overlaps end of linear array split into two block moves 

    if ((prd + sz) > pend) {
		copy_first = (pend-prd);

		memcpy(buf, prd, copy_first*sizeof(word));
		memcpy(&buf[copy_first], pstart, (sz-copy_first)*sizeof(word));
    }
    else
		memcpy(buf, prd, sz*sizeof(word));

    /* increment prd and wrap around if required */

    new_prd = prd + sz;
    if (new_prd >= pend)
		prd = pstart + (new_prd - pend);
    else
		prd = new_prd;

    assert(CheckFifo() == OK);
    return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoData::HowFull
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Determines number of used words in FIFO.

\*--------------------------------------------------------------------------*/

void FifoData::HowFull(USHORT *words)
//  word   *words;	number used words in FIFO			 
{
    assert(words != NULL);

    assert(CheckFifo() == OK);
    if (pwr >= prd)
		*words = pwr - prd;
    if (prd > pwr)
		*words = size - (prd - pwr);
}

/*-------------------------------------------------------------------------*\

	FUNCTION.: FifoData::CheckFifo
    AUTHOR...: David Rowe
	DATE.....: 19/11/97

    Performs a few sanity checks on DSP FIFO structure.  Returns OK is DSP
    FIFO checks out returns.

\*-------------------------------------------------------------------------*/

int FifoData::CheckFifo()
{
    if (pend < pstart) return(FAIL);
    if (pwr < pstart) return(FAIL);
    if (pwr > pend) return(FAIL);
    if (prd < pstart) return(FAIL);
    if (prd > pend) return(FAIL);

    return(OK);
}
